<?php

/**
 * Class BonicaXmlRpc
 *
 * @property array methods
 */
class BonicaXmlRpc extends wp_xmlrpc_server
{
    const BONICA_SITE = 'bonica.pro';
    const SCHEME_SITE = 'https';

    /**
     * Bonica_XMLRPC constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->methods = array_merge($this->methods, [
            'bonica.getCategories' => 'this:getCategories',
            'bonica.checkKey' => 'this:checkApiKey',
            'bonica.checkAuth' => 'this:checkAuth',
            'bonica.getPublishDate' => 'this:getPublishDate',
            'bonica.checkPostLocked' => 'this:checkPostLocked',
            'bonica.getTags' => 'this:getTags',
            'bonica.getPosts' => 'this:getPosts',
            'bonica.getPostsLocked' => 'this:getPostsLocked',
            'bonica.uploadPost' => 'this:uploadPost',
            'bonica.getInfo' => 'this:getInfo',
            'bonica.getUrls' => 'this:getUrls',
            'wp.getPosts' => 'this:getPosts',
            'wp.editPost' => 'this:editPost',
        ]);
    }

    public function getInfo($args)
    {

        $username = $args[1];
        return [
            'plugin' => get_plugin_data(dirname(__DIR__) . '/bonica-plugin.php'),
            'user' => $this->getUserByLogin($username),
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'post_max_size' => ini_get('post_max_size'),
        ];
    }

    protected function getUserByLogin($email)
    {
        return get_user_by('login', $email);
    }

    protected function getOrCreateUser($email)
    {
        if (!$user = $this->getUserByLogin($email)) {
            $uid = wp_create_user($email, 'password', $email);
            $user = new WP_User($uid);
            $user->add_role('contributor');
        }
        return $user;
    }

    /**
     * Переопределяем метод IXR_Server::call для последовательной передачи аргументов
     *
     * @param $methodname
     * @param $args
     * @return IXR_Error|mixed|null
     */
    function call($methodname, $args)
    {
        $key = array_shift($args);
        if (!$this->checkKey($key)) {
            sleep(3); // brutforce soft-filter
            return new IXR_Error(-32601, 'Access is denied. Check Bonica API key');
        }
        return parent::call($methodname, $args);
    }

    public function checkApiKey()
    {
        return BonicaDebug::collectDebugInfo();
    }

    /**
     * @param $key
     * @return array|int
     */
    public function checkKey($key)
    {
        if ($key == get_option(Bonica::BONICA_APIKEY)) {
            update_option(Bonica::BONICA_AUTH_SUCCESS, 1);
            return get_plugin_data(BONICA_PLUGIN_PATH . DIRECTORY_SEPARATOR . 'bonica-plugin.php');
        } else {
            return 0;
        }
    }

    protected static function extractSrc($content)
    {
        if (preg_match_all('#src\s*=\s*(?=[\'"]([^"^\']*)[\'"])#iU', $content, $matches)) {
            return array_unique($matches[1]);
        } else {
            return [];
        }
    }

    /**
     * @param $data
     * @param $postId
     * @param $images
     * @return mixed
     */
    protected function attachImagesToPost($data, $postId, &$images, $author = null)
    {
        $images = [];
        foreach (self::extractSrc($data['post_content']) as $link) {
            $host = parse_url($link, PHP_URL_HOST);
            Logger::info("Start download image {$link}", ['author' => $author]);
            $url = !$host ? self::SCHEME_SITE . '://' . self::BONICA_SITE . '/' . ltrim($link, '/') : $link;
            $response = self::mediaSideloadImage($url . '?plugin=1', $postId, $author ? ['post_author' => $author] : null);
            if (is_numeric($response)) {
                $images[] = ['id' => $response, 'url' => $url];
                if ($src = wp_get_attachment_url($response)) {
                    $query = phpQuery::newDocumentHTML($data['post_content']);
                    $query->find("[src='$link']")
                        ->addClass('wp-image-' . $response)
                        ->addClass('size-full')
                        ->attr('src', $src);
                    $data['post_content'] = $query->htmlOuter();
                    Logger::info("Success download $link");
                } else {
                    Logger::error("Fail get attach url for link $link", $response);
                }
            } else {
                Logger::error("Fail store media, $link", $response);
            }
        }
        return $data;
    }

    /***
     * @param array $args
     * @return array|bool|IXR_Error|true
     */
    public function editPost($args)
    {
        if (isset($args[4]['post_name'])) {
            $args[4]['post_name'] = (new \Cocur\Slugify\Slugify())->slugify($args[4]['post_name']);
        }

        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        $postId = (int)$args[3];
        $author = $this->getUserByLogin($args[1]);
        $args[4] = $this->attachImagesToPost($args[4], $postId, $images, isset($author->ID) ? $author->ID : null);


        $this->escape($args);

        $post_id = (int)$args[3];
        $content_struct = $args[4];

        $post = get_post($post_id, ARRAY_A);

        if (empty($post['ID'])) {
            //return new IXR_Error(404, __('Invalid post ID.'));
            return [
                'error_code' => '404',
                'error_message' => 'Invalid post ID.',
            ];
        }

        if (!empty($images)) {
            $this->setPostThumbnail([$post['ID'], current($images)['id']]);
        }

        $this->escape($post);
        $merged_content_struct = array_merge($post, $content_struct);

        $result = wp_update_post($merged_content_struct, true);
        if (is_numeric($result)) {
            MetaPlugin::setMeta($postId, $args[4]['meta_title'], $args[4]['meta_description']);
            MetaPlugin::setBonicaMeta($postId, $args[4]['meta_keywords']);
            return ['link' => get_permalink((int)$args[3])];
        } else {
            return [
                'error_code' => $result->get_error_code(),
                'error_message' => $result->get_error_message(),
            ];
        }
    }

    public function wp_newPost($args)
    {
        if (!$this->minimum_args($args, 4)) {
            return $this->error;
        }
        $original = $args;
        $this->escape($args);
        $username = $args[1];
        $content_struct = $args[3];
        if (!$user = $this->getUserByLogin($username)) {
            return $this->error;
        }
        $content_struct['post_author'] = $user->ID;
        $postId = wp_insert_post($content_struct, true);
        if (is_numeric($postId)) {
            $updateArgs = [$original[0], $original[1], $original[2], $postId, $original[3]];
            $this->editPost($updateArgs);
            MetaPlugin::setMeta($postId, $args[3]['meta_title'], $args[3]['meta_description']);
            MetaPlugin::setBonicaMeta($postId, $args[3]['meta_keywords']);
            return [
                'post_id' => $postId,
                'link' => get_permalink($postId)
            ];
        } else {
            return [
                'error_code' => $postId->get_error_code(),
                'error_message' => $postId->get_error_message(),
            ];
        }
    }


    /**
     *
     * @param $file
     * @param $post_id
     * @param null $desc
     * @param string $return
     * @return false|int|mixed|object|string|WP_Error
     */
    protected static function mediaSideloadImage($file, $post_id, $post_data = null, $return = 'id')
    {
        if (!empty($file)) {
            preg_match('/[^\?]+\.(jpe?g|jpe|gif|png)\b/i', $file, $matches);
            if (!$matches) {
                return new WP_Error('image_sideload_failed', __('Invalid image URL'));
            }
            $file_array = [];
            $file_array['name'] = basename($matches[0]);
            $file_array['tmp_name'] = download_url($file);
            if (is_wp_error($file_array['tmp_name'])) {
                return $file_array['tmp_name'];
            }
            $id = media_handle_sideload($file_array, $post_id, null, $post_data);
            if (is_wp_error($id)) {
                @unlink($file_array['tmp_name']);
                return $id;
            } elseif ($return === 'id') {
                return $id;
            }
            $src = wp_get_attachment_url($id);
        }
        if (!empty($src)) {
            if ($return === 'src') {
                return $src;
            }
            $alt = isset($desc) ? esc_attr($desc) : '';
            $html = "<img src='$src' alt='$alt' />";
            return $html;
        } else {
            return new WP_Error('image_sideload_failed');
        }
    }

    /**
     * @return array|int
     */
    public function getTags()
    {
        $tags_data = get_tags(['hide_empty' => false]);
        $tags = [];
        foreach ($tags_data as $tag) {
            $tags[$tag->term_id] = $tag->name;
        }
        return $tags;
    }

    /**
     * @return array
     */
    public function getPostsLocked()
    {
        $items = [];
        $posts = get_posts([
            'post_type' => 'post',
            'post_status' => 'draft',
        ]);
        foreach ($posts as $post) {
            $user = wp_check_post_lock($post->ID);
            if ($user) {
                $user_info = get_userdata($user);
                $item['post_status'] = 'Черновик';
                $item['post_title'] = $post->post_title;
                $item['display_name'] = $user_info->display_name;
                $items[] = $item;
            }
        }
        $posts = get_posts([
            'post_type' => 'post',
            'post_status' => 'pending',
        ]);
        foreach ($posts as $post) {
            $user = wp_check_post_lock($post->ID);
            if ($user) {
                $user_info = get_userdata($user);
                $item['post_status'] = 'На утверждении';
                $item['post_title'] = $post->post_title;
                $item['display_name'] = $user_info->display_name;
                $items[] = $item;
            }
        }
        return $items;
    }

    /**
     * @param $item
     * @return int|string
     */
    public function checkPostLocked($item)
    {
        $user = wp_check_post_lock($item['id']);
        if ($user) {
            $user_info = get_userdata($user);
            return $user_info->display_name;
        }
        return '';
    }

    /**
     * @return array
     */
    public function getPosts($args)
    {
//        $posts = parent::wp_getPosts($args);
        $posts = $this->wp_getPosts($args);
        foreach ($posts as $key => &$post) {
            if (isset($args[3]['link'])) {
                $id = url_to_postid($args[3]['link']);
                if ($post['post_id'] != $id) {
                    unset($posts[$key]);
                    continue;
                }
            }
            $post['seo'] = [
                'title' => MetaPlugin::getTitle($post['post_id']),
                'description' => MetaPlugin::getDescription($post['post_id']),
                'keywords' => MetaPlugin::getKeywords($post['post_id']),
            ];
            $post_categories = wp_get_post_categories($post['post_id']);
            foreach ($post_categories as $c) {
                $cat = get_category($c);
                $post['category'] = $cat->name;
                $post['term_id'] = $cat->term_id;
            }
        }
        return $posts;
    }

    /**
     * @return array
     */
    public function getUrls($args)
    {
        $ids = get_posts(['post_type' => 'post', 'numberposts' => -1, 'fields' => 'ids']);
        return array_map(function ($id) {
            return get_permalink($id);
        }, $ids);
    }

    /**
     * @param array $args
     * @return array
     */
    public function wp_getPost($args)
    {
        if (!is_numeric($args[3])) {
            $args[3] = url_to_postid($args[3]);
        }

        if ($post = get_post($args[3], ARRAY_A)) {
            if ($post['post_status'] == 'trash') {
                return null;
            }
            $post['link'] = get_permalink($post['ID']);
            $advancedFields = [
                'seo' => [
                    'title' => MetaPlugin::getTitle($post['ID']),
                    'description' => MetaPlugin::getDescription($post['ID']),
                    'keywords' => MetaPlugin::getKeywords($post['ID']),
                ],
            ];
            return array_merge($post, $advancedFields);
        } else {
            return null;
        }
    }

    /**
     * @param $item
     * @return int|string
     */
    public function getPublishDate($item)
    {
        $post = get_post($item['id']);
        if ($post->post_status == 'publish') {
            return $post->post_date;
        } else {
            return '';
        }
    }

    public function checkAuth($args)
    {
        return $this->login($args[1], $args[2]);
    }

    /**
     * @return array|int
     */
    public function getCategories()
    {
        return get_categories(['hide_empty' => false]);
    }

    public function wp_getPosts($args)
    {
        if (!$this->minimum_args($args, 3))
            return $this->error;
        $this->escape($args);
        $filter = isset($args[3]) ? $args[3] : [];

        if (isset($args[4])) {
            $fields = $args[4];
        } else {
            /** This action is documented in wp-includes/class-wp-xmlrpc-server.php */
            $fields = apply_filters('xmlrpc_default_post_fields', ['post', 'terms', 'custom_fields'], 'wp.getPosts');
        }

        /** This action is documented in wp-includes/class-wp-xmlrpc-server.php */
        do_action('xmlrpc_call', 'wp.getPosts');

        $query = [];

        if (isset($filter['post_type'])) {
            $post_type = get_post_type_object($filter['post_type']);
            if (!((bool)$post_type))
                return new IXR_Error(403, __('Invalid post type.'));
        } else {
            $post_type = get_post_type_object('post');
        }

        /*if ( ! current_user_can( $post_type->cap->edit_posts ) )
            return new IXR_Error( 401, __( 'Sorry, you are not allowed to edit posts in this post type.' ) );*/

        $query['post_type'] = $post_type->name;

        if (isset($filter['post_status'])) {
            $query['post_status'] = $filter['post_status'];
        }

        if (isset($filter['number'])) {
            $query['numberposts'] = absint($filter['number']);
        }

        if (isset($filter['offset']))
            $query['offset'] = absint($filter['offset']);

        if (isset($filter['paged']))
            $query['paged'] = absint($filter['paged']);

        if (isset($filter['orderby'])) {
            $query['orderby'] = $filter['orderby'];

            if (isset($filter['order']))
                $query['order'] = $filter['order'];
        }

        if (isset($filter['s'])) {
            $query['s'] = $filter['s'];
        }

        $defaults = [
            'numberposts' => -1,
            'category' => 0,
            'orderby' => 'date',
            'order' => 'DESC',
            'include' => [],
            'exclude' => [],
            'meta_key' => '',
            'meta_value' => '',
            'post_type' => 'post',
            'suppress_filters' => true
        ];

        $r = wp_parse_args($query, $defaults);
        if (empty($r['post_status']))
            $r['post_status'] = 'publish';
        if (!empty($r['numberposts']) && empty($r['posts_per_page']))
            $r['posts_per_page'] = $r['numberposts'];
        if (!empty($r['category']))
            $r['cat'] = $r['category'];
        if (!empty($r['include'])) {
            $incposts = wp_parse_id_list($r['include']);
            $r['posts_per_page'] = count($incposts);  // only the number of posts included
            $r['post__in'] = $incposts;
        } elseif (!empty($r['exclude']))
            $r['post__not_in'] = wp_parse_id_list($r['exclude']);

//        $r['ignore_sticky_posts'] = false;
//        $r['no_found_rows'] = true;

        $get_posts = new WP_Query;
        $posts_list = $get_posts->query($r);

        foreach ($posts_list as $key => $result) {
            $posts_list[$key] = get_object_vars($result);
        }

        if (!$posts_list)
            return [];

        // Holds all the posts data.
        $struct = [];

        foreach ($posts_list as $post) {

            $struct[] = $this->_prepare_post($post, $fields);
        }

        return $struct;
    }

    /**
     * @param $args
     * @param false $force
     * @return bool|int|null
     */
    public function setPostThumbnail($args, $force = false)
    {
        if ($force || !get_the_post_thumbnail($args[0])) {
            return set_post_thumbnail($args[0], $args[1]);
        }
        return null;
    }
}
